<?php

if (count(get_included_files()) == 1) {
	define('__name__', "__main__");
} else {
	define('__name__', "x");
}

function log_debug()
{
	$args = func_get_args();
	$fmt = $args[0];
	list($b, $b2) = debug_backtrace();
	$bt = basename($b["file"]).":".$b["line"].":".$b2["function"].": ";
	$args[0] = $bt;
	return vfprintf(STDERR, "%s".$fmt."\n", $args);
}
function log_debug_r($mix)
{
	ob_start();
	print_r($mix);
	$debug_msg = ob_get_clean();
	list($b, $b2) = debug_backtrace();
	$bt = basename($b["file"]).":".$b["line"].":".$b2["function"].": ";
	return fprintf(STDERR, "%s[%s]\n", $bt, $debug_msg);
}

if (false) {
	$lib_prefix = '/mnt/source/libsynobackup/lib/s3common/aws-sdk';
	define('AWS_PHAR', true);

	require_once $lib_prefix.'/Symfony/Component/ClassLoader/UniversalClassLoader.php';

	$classLoader = new Symfony\Component\ClassLoader\UniversalClassLoader();
	$classLoader->registerNamespaces(array(
	    'Aws'      => $lib_prefix,
	    'Guzzle'   => $lib_prefix,
	    'Symfony'  => $lib_prefix,
	    'Doctrine' => $lib_prefix,
	    'Monolog'  => $lib_prefix,
	));
	$classLoader->register();
} else {
	$aws_sdk_path = getenv("AWS_SDK_PATH");
	if (FALSE == $aws_sdk_path) {
		require '/usr/syno/aws/aws.phar';
	} else {
		require $aws_sdk_path;
	}
}

use Aws\S3\S3Client;
use Guzzle\Common\Event;
use Guzzle\Http\Exception\CurlException;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

// because AWS time stamp use UTC time
date_default_timezone_set("UTC");

function _convert_exception($e)
{
	try {
		log_debug($e);
	} catch(Exception $ee) {
		; // pass
	}

	if (is_subclass_of($e, 'Aws\Common\Exception\ServiceResponseException')) {
		// http://docs.aws.amazon.com/AmazonS3/latest/API/ErrorResponses.html
		return array(
			"success" => false,
			"error_class" => get_class($e),
			"error_message" => $e->getMessage(),
			"http_status_code" => $e->getStatusCode(),
			"aws_error_type" => $e->getExceptionType(),
			"aws_error_code" => $e->getExceptionCode(),
		);
	} elseif (is_subclass_of($e, 'Guzzle\Http\Exception\CurlException')) {
		// Aws\Common\Exception\TransferException extend CurlException>RequestException>
		return array(
			"success" => false,
			"error_class" => get_class($e),
			"error_message" => $e->getMessage(),
			"curl_message" => $e->getError(),
			"curl_error_code" => $e->getErrorNo(),
		);
	} else {
		return array(
			"success" => false,
			"error_class" => get_class($e),
			"error_message" => $e->getMessage(),
		);
	}
}
function _convert_time($time_str)
{
	// 2014-06-17T10:30:51.000Z
	$time_str = substr($time_str, 0, 19);
	$ts = strptime($time_str, '%FT%T');
	$tt = mktime($ts['tm_hour'], $ts['tm_min'], $ts['tm_sec'],
		$ts['tm_mon']+1, $ts['tm_mday'], ($ts['tm_year'] + 1900));

	return $tt;
}
function _convert_time2($time_str)
{
	// Thu, 14 Aug 2014 08:58:50 GMT
	$ts = strptime($time_str, '%a, %e %b %Y %T %Z');
	$tt = mktime($ts['tm_hour'], $ts['tm_min'], $ts['tm_sec'],
		$ts['tm_mon']+1, $ts['tm_mday'], ($ts['tm_year'] + 1900));

	return $tt;
}

class ServerTerminateException extends Exception {};
class SynoCommandPipeIO
{
	function read_int()
	{
		$n = 4;
		while ($n > 0) {
			if (false == ($read_data = fread(STDIN, $n))) {
				if (feof(STDIN)) {
					throw new ServerTerminateException();
				}
				throw new Exception("read failed");
			}
			$data .= $read_data;
			$n -= mb_strlen($read_data, '8bit');
		}

		if ("" === $data && feof(STDIN)) {
			throw new ServerTerminateException();
		}
		$unpack_data = unpack("I", $data);
		return $unpack_data[1];
	}
	function read_string()
	{
		$len = $this->read_int();

		if (0 === $len) {
			return "";
		}
		$n = $len;
		while ($n > 0) {
			$to_read = 4096 > $n ? $n : 4096;
			if (FALSE == ($read_str = fread(STDIN, $to_read))) {
				throw new Exception("read failed");
			}
			$str .= $read_str;
			$n -= mb_strlen($read_str, '8bit');
			if (feof(STDIN) && $n > 0) {
				throw new Exception("read eof");
			}
		}
		return $str;
	}
	function read_json()
	{
		$str = $this->read_string();

		return json_decode($str, true);
	}

	function write_int($val)
	{
		$pack_data = pack("I", $val);
		fwrite(STDOUT, $pack_data, 4);
	}
	function write_string($msg)
	{
		$len = strlen($msg);

		$this->write_int($len);
		fwrite(STDOUT, $msg, $len);
	}
	function write_json($json)
	{
		$this->write_string(json_encode($json));
		fflush(STDOUT);
	}
	function write_exception($e)
	{
		$this->write_json(_convert_exception($e));
	}
}

class SynoS3Listener implements EventSubscriberInterface
{
	private $progress_listener = null;

	public static function getSubscribedEvents()
	{
		return array(
			'command.before_send' => 'onBeforeSend',
			'client.create_request' => 'onCreateRequest',
			'curl.callback.progress' => 'onCurlProgress',
		);
	}
	public function onBeforeSend(Event $event)
	{
		$command = $event['command'];
		$request = $command->getRequest();
		$userAgent = (string) $request->getHeader('User-Agent');
		$request->setHeader('User-Agent', getenv("SYNO_USER_AGENT") . " " . $userAgent);
	}


	public function setProgressListener($fn)
	{
		$this->progress_listener = $fn;
	}
	public function removeProgressListener()
	{
		$this->progress_listener = null;
	}
	public function onCreateRequest(Event $event)
	{
		$request = $event['request'];

		if ($this->progress_listener !== null) {
			$request->getCurlOptions()->set("progress", true);
		}
	}
	public function onCurlProgress(Event $event)
	{
		if ($this->progress_listener !== null) {
			call_user_func_array($this->progress_listener, array($event));
		}
	}
}
class SynoS3Wrapper
{
	private $client = null;
	private $s3api = null;
	private $s3listener = null;
	private $io = null;

	function __construct($aws_config, $io)
	{
		$this->io = $io;
		$this->s3listener = new SynoS3Listener();
		$this->createS3Client($aws_config);
		//$this->_init_api_map();
	}

	function createS3Client($aws_config)
	{
		log_debug("\033[34minit_client\033[0m");
		$this->client = S3Client::factory($aws_config);
		$this->client->addSubscriber($this->s3listener);
	}
	function _init_api_map()
	{
		$this->s3api = array(
			"completeMultipartUpload" => true,
		);

		$s3api = require('phar://aws.phar/Aws/S3/Resources/s3-2006-03-01.php');

		foreach ($s3api as $cmd => $value) {
			$this->s3api["$cmd"] = true;
		}
	}

	private $prev_uplodaed_size = 0;
	private $last_update_time = 0;

	function enableUploadProgress()
	{
		$this->last_update_time = time();
		$this->prev_uploaded_size = 0;
		$this->s3listener->setProgressListener(array($this, 'onUploadProgress'));
	}
	function disableUploadProgress()
	{
		$this->s3listener->removeProgressListener();
		$this->prev_uploaded_size = 0;
		$this->last_update_time = 0;
	}
	function onUploadProgress(Event $event)
	{
		if ($this->prev_uploaded_size >= $event["uploaded"]) {
			// ignore insane and no information case
			return ;
		}

		$this->prev_uploaded_size = $event["uploaded"];

		if (!$this->io || time() - $this->last_update_time < 3) {
			return ;
		}

		$this->io->write_json(array(
			"success" => true,
			"complete" => false,
			"uploaded" => $this->prev_uploaded_size,
		));

		$this->last_update_time = time();
	}

	function getVersion($in_json)
	{
		return array(
			"success" => true,
			"version" => Aws\Common\Aws::VERSION,
		);
	}
	function getRegions($in_json)
	{
		$res = $this->client->getRegions();

		$out_json = array(
			"success" => true,
			"Regions" => array()
		);

		foreach ($res as $key => $value) {
			$out_json["Regions"][] = $key;
		}

		return $out_json;
	}
	function setRegion($region)
	{
		$this->client->setRegion($region);
		return true;
	}
	function getRegion()
	{
		return $this->client->getRegion();
	}

	function isValidBucketName($in_json)
	{
		return array(
			"success" => $this->client->isValidBucketName($in_json["Bucket"])
		);
	}
	function createBucket($in_json)
	{
		try {
			$res = $this->client->CreateBucket($in_json);
		} catch (Exception $e1) {
			// DSM #69035 - workaround for frankfurt problem.
			$region_org = $this->getRegion();
			$this->setRegion("us-west-1");
			try {
				$res = $this->getBucketLocation(array("Bucket" => $in_json["Bucket"]));
			} catch (Exception $e2) {
				$this->setRegion($region_org);
				throw($e1);
			}
			$this->setRegion($region_org);
			if ($res["Location"] === "eu-central-1") {
				$new_e = new Aws\S3\Exception\BucketAlreadyOwnedByYouException("Your previous request to create the named bucket succeeded and you already own it.");
				$new_e->setResponse(new Guzzle\Http\Message\Response("409"));
				throw($new_e);
			}
			throw($e1);
		}
		return array("success" => true);
	}
	function headBucket($in_json)
	{
		$res = $this->client->HeadBucket($in_json);
		return array("success" => true);
	}
	function listBuckets($in_json)
	{
		$res = $this->client->ListBuckets($in_json);
		$out_json = array(
			"success" => true,
			"Buckets" => array()
		);
		foreach ($res->get("Buckets") as $rec) {
			$out_json["Buckets"][] = $rec["Name"];
		}
		return $out_json;
	}
	function getBucketLocation($in_json)
	{
		$res = $this->client->getBucketLocation($in_json);

		return array(
			"success" => true,
			"Location" => $res->get("Location")
		);
	}

	function putObject($in_json)
	{
		if (preg_match('/[\x00-\x1F\x7F]/', $in_json["Key"])) {
			log_debug("key contains control char. (%s)", $in_json["Key"]);
			return array(
				"success" => false,
				"error_class" => "PhpPathContainCtrlChar",
				"error_message" => "path contains control chars"
			);
		}
		if (false === ($handle = fopen($in_json["Body"], 'r'))) {
			throw Exception("can not open file [".$in_json["Body"]."]");
		}
		$in_json["Body"] = $handle;

		try {
			$this->enableUploadProgress();
			$res = $this->client->PutObject($in_json);
			$this->disableUploadProgress();

			if (is_resource($handle)) {
				fclose($handle);
			}

			return array(
				"success" => true,
				"RequestId" => $res->get("RequestId"),
				"ETag" => $res->get("ETag")
			);
		} catch (Exception $e) {
			$this->disableUploadProgress();
			if (is_resource($handle)) {
				fclose($handle);
			}
			throw $e;
		}
	}
	function listObjects($in_json)
	{
		$res = $this->client->ListObjects($in_json);
		$prefix_len = strlen($in_json["Prefix"]);
		$out_json = array(
			"success" => true,
			"count" => 0,
		);
		if ($res->get("NextMarker")) {
			$out_json["NextMarker"] = $res->get("NextMarker");
		} elseif ($res->get("IsTruncated")) {
			$out_json["NextMarker"] = $res->get("Contents")[count($res->get("Contents")) - 1]["Key"];
		}
		if ($res->get("CommonPrefixes")) {
			$out_json["folder"] = array();
			foreach ($res->get("CommonPrefixes") as $rec) {
				$out_json["folder"][] = array(
					"Name" => substr($rec["Prefix"], $prefix_len),
				);
				$out_json["count"] += 1;
			}
		}
		if ($res->get("Contents")) {
			$out_json["file"] = array();
			foreach ($res->get("Contents") as $rec) {
				if (strlen($rec["Key"]) <= $prefix_len) {
					continue;
				}
				$out_json["file"][] = array(
					"Name" => substr($rec["Key"], $prefix_len),
					"ETag" => $rec["ETag"],
					"LastModified" => _convert_time($rec["LastModified"]),
					"ContentLength" => $rec["Size"],
				);
				$out_json["count"] += 1;
			}
		}
		return $out_json;
	}
	function headObject($in_json)
	{
		$res = $this->client->HeadObject($in_json);

		return array(
			"success" => true,
			"LastModified" => _convert_time2($res->get("LastModified")),
			"ETag" => $res->get("ETag"),
			"ContentLength" => $res->get("ContentLength")
		);
	}
	function deleteObject($in_json)
	{
		$res = $this->client->DeleteObject($in_json);
		return array(
			"success" => true,
			"RequestId" => $res->get("RequestId"),
		);
	}
	function getObject($in_json)
	{
		$res = $this->client->GetObject($in_json);

		return array(
			"success" => true,
			"ETag" => $res->get("ETag"),
			"RequestId" => $res->get("RequestId")
		);
	}
	function copyObject($in_json)
	{
		$res = $this->client->CopyObject($in_json);

		return array(
			"success" => true,
			"RequestId" => $res->get("RequestId")
		);
	}

	function createMultipartUpload($in_json)
	{
		if (preg_match('/[\x00-\x1F\x7F]/', $in_json["Key"])) {
			log_debug("key contains control char. (%s)", $in_json["Key"]);
			return array(
				"success" => false,
				"error_class" => "PhpPathContainCtrlChar",
				"error_message" => "path contains control chars"
			);
		}
		$res = $this->client->CreateMultipartUpload($in_json);

		return array(
			"success" => true,
			"UploadId" => $res->get("UploadId")
		);
	}
	function uploadPart($in_json)
	{
		if (false === ($handle = fopen($in_json["Body"], 'r'))) {
			throw Exception("can not open file [".$in_json["Body"]."]");
		}
		$in_json["Body"] = $handle;

		try {
			$this->enableUploadProgress();
			$res = $this->client->UploadPart($in_json);
			$this->disableUploadProgress();

			if (is_resource($handle)) {
				fclose($handle);
			}

			return array(
				"success" => true,
				"RequestId" => $res->get("RequestId"),
				"ETag" => $res->get("ETag")
			);
		} catch (Exception $e) {
			$this->disableUploadProgress();
			if (is_resource($handle)) {
				fclose($handle);
			}
			throw $e;
		}
	}
	function completeMultipartUpload($in_json)
	{
		try {
			$res = $this->client->CompleteMultipartUpload($in_json);
		} catch (Guzzle\Common\Exception\RuntimeException $e) {
			return array(
				"success" => true
			);
		}

		return array(
			"success" => true,
			"ETag" => $res->get("ETag"),
			"RequestId" => $res->get("RequestId")
		);
	}
	function abortMultipartUpload($in_json)
	{
		$res = $this->client->abortMultipartUpload($in_json);

		return array("success" => true);
	}
	function listMultipartUploads($in_json)
	{
		$res = $this->client->listMultipartUploads($in_json);

		$out_json = array(
			'success' => true,
			'uploads' => array(),
		);

		if (!$res->get("Uploads")) {
			return $out_json;
		}
		foreach ($res->get("Uploads") as $rec) {
			$out_json['uploads'][] = array(
				"Key" => $rec["Key"],
				"UploadId" => $rec["UploadId"],
			);
		}

		return $out_json;
	}
	function listParts($in_json)
	{
		$res = $this->client->listParts($in_json);

		$out_json = array(
			'success' => true,
			'parts' => array()
		);

		if ($res->get("IsTruncated")) {
			$out_json['NextPartNumberMarker'] = $res->get('NextPartNumberMarker');
		}

		if ($res->get("Parts")) {
			foreach ($res->get("Parts") as $rec) {
				$out_json['parts'][] = array(
					'PartNumber' => $rec['PartNumber'],
					'LastModified' => $rec['LastModified'],
					'ETag' => $rec['ETag'],
					'Size' => $rec['Size'],
				);
			}
		}

		return $out_json;
	}
}

function check_need_redirect($wrapper, $aws_config, $in_json)
{
	if ((array_key_exists("region", $aws_config) && false !== $aws_config["region"])
	 || (array_key_exists("base_url", $aws_config) && false !== $aws_config["base_url"])) {
		// only support one region for this process live cycle
		return false;
	}
	if (!array_key_exists("Bucket", $in_json)) {
		return false;
	}
	if (false !== strpos($in_json["fn"], "Bucket")) {
		return false;
	}

	log_debug("\033[33mcheck bucket region\033[0m");
	$res = $wrapper->getBucketLocation(array("Bucket" => $in_json["Bucket"]));

	if ($res["Location"] === "EU") {
		return "eu-west-1";
	} else {
		return $res["Location"];
	}
}

function _main()
{
	$io = new SynoCommandPipeIO();

	try {
		$aws_config = array(
			'key' => getenv("AWS_ACCESS_KEY_ID"),
			'secret' => getenv("AWS_SECRET_ACCESS_KEY"),
			'base_url' => getenv("AWS_URL"),
			'region' => getenv("AWS_REGION"),
			'scheme' => getenv("AWS_SCHEME") ? getenv("AWS_SCHEME") : "http",
			'ssl.certificate_authority' => '/usr/syno/aws/cacert.pem',
			'curl.options' => array(
				'CURLOPT_CONNECTTIMEOUT' => 60,
				'CURLOPT_LOW_SPEED_LIMIT' => 1,
			       	'CURLOPT_LOW_SPEED_TIME' => 1800,
				'CURLOPT_SSLVERSION' => 'CURL_SSLVERSION_TLSv1',
				'CURLOPT_VERBOSE' => getenv("AWS_DEBUG"),
			),
		);
		$wrapper = new SynoS3Wrapper($aws_config, $io);
	} catch (Exception $e) {
		$io->write_exception($e);
		return 1;
	}

	log_debug("\033[33mstart\033[0m");
	$io->write_string("start");

	while (true) {
		try {
			$in_json = $io->read_json();

			if ($in_json === null) {
				$resp = array("success" => false);
				if (JSON_ERROR_UTF8 === json_last_error()) {
					$resp["error_class"] = "PhpPathContainNonUTF8";
					$resp["error_message"] = "path contains non-utf8 chars";
				} else if (JSON_ERROR_CTRL_CHAR === json_last_error()) {
					$resp["error_class"] = "PhpPathContainCtrlChar";
					$resp["error_message"] = "path contains control chars";
				}
				log_debug("finished with error. (%s)", json_encode($resp));
				$io->write_json($resp);
				continue;
			}

			$region = check_need_redirect($wrapper, $aws_config, $in_json);
			if (false !== $region) {
				$aws_config["region"] = $region;
				$wrapper->createS3Client($aws_config);
			}

			$fn = $in_json["fn"];

			unset($in_json["fn"]);

			log_debug("\033[33mexec $fn(%s)\033[0m", json_encode($in_json));
			$res = call_user_func_array(array($wrapper, $fn), array($in_json));
			log_debug("finished");

			$io->write_json($res);
		} catch (ServerTerminateException $e) {
			break;
		} catch (CurlException $e) {
			$wrapper->createS3Client($aws_config);
			$io->write_exception($e);
			continue;
		} catch (\InvalidArgumentException $e) {
			$io->write_exception($e);
			continue;
		} catch (Exception $e) {
			if (0 == strcmp($e->getMessage(), "read failed")) {
				continue;
			}
			$io->write_exception($e);
			continue;
		}
	}

	log_debug("aws_agent terminate");
	return 0;
}

if (__name__ == "__main__") {
	_main();
}
